<?php

declare(strict_types=1);

namespace Juweliere\UserBundle\Security;

use Juweliere\JuwApiClient\Entity\User\Login;
use GuzzleHttp\Exception\ClientException;
use Juweliere\ApiBundle\Service\JuwApi;
use Juweliere\UserBundle\Controller\LoginController;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\Exception\CustomUserMessageAuthenticationException;
use Symfony\Component\Security\Core\Exception\InvalidCsrfTokenException;
use Symfony\Component\Security\Csrf\CsrfToken;
use Symfony\Component\Security\Csrf\CsrfTokenManagerInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationFailureHandlerInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationSuccessHandlerInterface;
use Symfony\Component\Security\Http\Authenticator\FormLoginAuthenticator;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\RememberMeBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\UserBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Passport;
use Symfony\Component\Security\Http\Authenticator\Passport\SelfValidatingPassport;
use Symfony\Component\Security\Http\HttpUtils;
use Symfony\Component\Security\Http\Util\TargetPathTrait;

class LoginFormAuthenticator extends FormLoginAuthenticator
{
    use TargetPathTrait;

    public const string LOGIN_ROUTE = 'juweliere_user_user_login';

    public const string AUTHENTICATION_ERROR = '_security.last_error';

    private UserProvider $userProvider;

    public function __construct(
        HttpUtils                                  $httpUtils,
        AuthenticationSuccessHandlerInterface      $successHandler,
        AuthenticationFailureHandlerInterface      $failureHandler,
        UserProvider                               $userProvider,
        private readonly UrlGeneratorInterface     $urlGenerator,
        private readonly JuwApi                    $juwApi,
        private readonly LoggerInterface           $logger,
        private readonly RequestStack              $requestStack,
        private readonly CsrfTokenManagerInterface $csrfTokenManager
    )
    {
        $options = ["use_forward" => false, "username_parameter" => "username", "username_password" => "password"];

        parent::__construct($httpUtils, $userProvider, $successHandler, $failureHandler, $options);
        $this->userProvider = $userProvider;
    }

    #[\Override]
    public function supports(Request $request): bool
    {
        return self::LOGIN_ROUTE === $request->attributes->get('_route')
            && $request->isMethod('POST');
    }

    #[\Override]
    public function authenticate(Request $request): Passport
    {
        $token = new CsrfToken('authenticate', $request->get('_csrf_token'));

        if (!$this->csrfTokenManager->isTokenValid($token)) {
            throw new InvalidCsrfTokenException();
        }

        $login = new Login();
        $login->setUsername( $request->get('username') );
        $login->setPassword( $request->get('password'));


        try {
            $userApi = $this->juwApi->users->login($login);
        } catch (ClientException $e) {
            throw new CustomUserMessageAuthenticationException('Invalid credentials: ' . $e->getCode());
        } catch (\Throwable $e) {
            $this->logger->critical("Login Form Error: " . $e->getMessage());
            throw new CustomUserMessageAuthenticationException('Internal error.');
        }

        $user = $this->userProvider->mapApiUser($userApi);

        return new SelfValidatingPassport(
            new UserBadge($user->getId(), $this->userProvider->loadUserByIdentifier(...)),
            [new RememberMeBadge()]
        );
    }

    #[\Override]
    public function onAuthenticationSuccess(Request $request, TokenInterface $token, $firewallName): RedirectResponse
    {   $session = $this->requestStack->getSession();

        if ($request->get("isXHR")) {
            $redirect = $this->urlGenerator->generate("juweliere_user_user_login_status");
        } elseif (!is_null($request->get("_target_path"))) {
            $redirect = $request->get("_target_path");
        }
        elseif ($redirect = $session->get(LoginController::SESSION_LOGIN_ORIGIN)) {
            $session->remove(LoginController::SESSION_LOGIN_ORIGIN);
        }
        elseif ($redirect = $session->get(LoginController::SESSION_LOGIN_REDIRECT)) {
            $session->remove(LoginController::SESSION_LOGIN_REDIRECT);
        }
        elseif ($targetPath = $this->getTargetPath($request->getSession(), $firewallName)) {
            $redirect = $targetPath;
        }

        if (is_null($redirect)) {
            $redirect = $this->urlGenerator->generate('juweliere_user_user_profile_edit');
        }

        return new RedirectResponse($redirect);
    }

    #[\Override]
    public function onAuthenticationFailure(Request $request, AuthenticationException $exception): Response
    {
        $request->getSession()->set("Error", $exception);

        $redirect = $this->urlGenerator->generate(self::LOGIN_ROUTE);
        if ($request->get("isXHR")) {
            $redirect = $this->urlGenerator->generate("juweliere_user_user_login_status");
        }

        return new RedirectResponse($redirect);
    }

    #[\Override]
    protected function getLoginUrl(Request $request): string
    {
        return $this->urlGenerator->generate(self::LOGIN_ROUTE);
    }
}
