<?php

declare(strict_types=1);

namespace Juweliere\UserBundle\Security;

use GuzzleHttp\Exception\GuzzleException;
use Juweliere\ApiBundle\Service\JuwApi;
use Juweliere\JuwApiClient\Entity\User as JuweliereApiUser;
use Juweliere\UserBundle\Entity\JuweliereUser;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Security\Core\Exception\UnsupportedUserException;
use Symfony\Component\Security\Core\User\PasswordUpgraderInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;

class UserProvider implements UserProviderInterface, PasswordUpgraderInterface
{
    const string SESSION_KEY_REFRESH_USER = "refreshUser";

    private ?JuweliereUser $user = null;

    /**
     * UserProvider constructor.
     *
     * @param JuwApi       $juwApi
     * @param RequestStack $requestStack
     */
    public function __construct(private readonly JuwApi $juwApi, private readonly RequestStack $requestStack)
    {
    }

    /**
     * The loadUserByIdentifier() method was introduced in Symfony 5.3.
     * In previous versions it was called loadUserByUsername()
     *
     * Symfony calls this method if you use features like switch_user
     * or remember_me. If you're not using these features, you do not
     * need to implement this method.
     *
     *
     * @throws GuzzleException
     */
    #[\Override]
    public function loadUserByIdentifier(string $identifier): UserInterface
    {
        if (is_null($this->user)) {
            $apiUser = $this->juwApi->users->get($identifier);

            return $this->mapApiUser($apiUser);
        }

        return $this->user;
    }

    /**
     * Refreshes the user after being reloaded from the session.
     *
     * When a user is logged in, at the beginning of each request, the
     * User object is loaded from the session and then this method is
     * called. Your job is to make sure the user's data is still fresh by,
     * for example, re-querying for fresh User data.
     *
     * If your firewall is "stateless: true" (for a pure API), this
     * method is not called.
     *
     * @throws GuzzleException
     */
    #[\Override]
    public function refreshUser(UserInterface $user): UserInterface
    {
        if (!$user instanceof JuweliereUser) {
            throw new UnsupportedUserException(sprintf('Invalid user class "%s".', $user::class));
        }

        $session = $this->requestStack->getSession();
        if ($session->get(self::SESSION_KEY_REFRESH_USER) === 1) {

            $apiUser = $this->juwApi->users->get($user->getId());
            $session->remove(self::SESSION_KEY_REFRESH_USER);

            return $this->mapApiUser($apiUser);
        }

        return $user;
    }


    public function mapApiUser(JuweliereApiUser $userApi): ?JuweliereUser
    {
        $this->user = new JuweliereUser();

        $this->user->setId($userApi->getId());
        $this->user->setEmail($userApi->getEmail());
        $this->user->setGender($userApi->getGender());
        $this->user->setFirstName($userApi->getFirstName());
        $this->user->setLastName($userApi->getLastName());
        $this->user->setAddresses($userApi->getAddresses());
        $this->user->setWishlistItems($userApi->getWishlistItems());
        $this->user->setContactPhone($userApi->getContactPhone());
        $this->user->setContactEmail($userApi->getContactEmail());
        $this->user->setBirthdate($userApi->getBirthdate());
        $this->user->setPhone($userApi->getPhone());

        return $this->user;
    }

    /**
     * Tells Symfony to use this provider for this User class.
     */
    #[\Override]
    public function supportsClass($class): bool
    {
        return JuweliereUser::class === $class || is_subclass_of($class, JuweliereUser::class);
    }

    /**
     * Upgrades the hashed password of a user, typically for using a better hash algorithm.
     */
    #[\Override]
    public function upgradePassword(PasswordAuthenticatedUserInterface $user, string $newHashedPassword): void
    {
        // TODO: when hashed passwords are in use, this method should:
        // 1. persist the new password in the user storage
        // 2. update the $user object with $user->setPassword($newHashedPassword);
    }

    public function loadUserByUsername($username): void
    {
        // TODO: Implement loadUserByUsername() method.
    }

    public function setUser(JuweliereUser $user): void
    {
        $this->user = $user;
    }
}
